/** sendsms: command line utility to send sms messages using gsm phone/modem.
* Compilation:  g++ -o sendsms sendsms.cpp
* \author L.A.M.P.
* \version V1.0 - 2012.02.05
* \version V2.0 - 2012.10.24: 	Potential message length problems was fixed.
*								Delay between commands diminished.
*								Command execution status is now checked.
*								Char* to string conversion warnings during compilation fixed.
*								Debug (verbose) exec mode implemented.
*                               Message Center Number definition implemented.
*                               3G chip pin number definition implemented.
*                               The validity period of the message can now be set.
*/

// For general help see
// http://www.developershome.com/sms/cscaCommand.asp
// http://www.tomshardware.com/forum/47165-39-commands-cmgs-quot-send-short-message-quot
// http://www.shapeshifter.se/2008/04/30/list-of-at-commands/
// http://www.smssolutions.net/tutorials/gsm/sendsmsat/

#define VERSION "V1.1 - 2012.10.24"
#define MAX_STR_LEN 1024
#define MAX_MSG_LEN 255         // Max SMS message length

// C includes
#include <getopt.h>		/* getopt_long() */
#include <stdio.h>		/* Standard input/output definitions */
#include <stdlib.h>		/* Standard library definitions */
#include <string.h>		/* String function definitions */
#include <unistd.h>		/* UNIX standard function definitions */
#include <fcntl.h>		/* File control definitions */
#include <errno.h>		/* Error number definitions */
#include <termios.h>	/* POSIX terminal control definitions */

class Modem3G
{
	private:

		struct termios _newOptions;
		struct termios _oldOptions;
		int _bRate;
		char _port[MAX_STR_LEN+1];
		char _sca[MAX_STR_LEN+1];
		int _fd;
		int _oldFlags;
		int _debug;
		char _response[MAX_STR_LEN+1];

        int _sendCmd (char* cmdIn)
		{
			char cr[2];
			char r;
			int i, j;
            int status = 0;
            char* okPtr;

			cr[0] = '\015';
			cr[1] = '\000';

			_response[0] = '0';

            if (_debug)
                printf("Enviando \"%s\": ", cmdIn);

            if (write(_fd, cmdIn, strlen(cmdIn)) != strlen(cmdIn)) {	// send command
                strcpy(_response, "ERRO: no pude enviar comando ao modem");
                status = -1;
			}

            else {

                if (write(_fd, cr, strlen(cr)) != strlen(cr)) {	        // send cr
                    strcpy(_response, "ERRO: no pude enviar CR ao modem");
                    status = -1;
                }

                else {

                    usleep(500000);

                    // get modem response

                    for (i=0, j=0; (j < 200) && (i == 0); j++) {
                        i = read(_fd,&r,1);
                    }

                    if (i == 0) {
                        strcpy(_response, "ERRO: nenhuma resposta do modem ao comando");
                        status = -1;
                    }

                    else {

                        i = 0;
                        if ((r >= 32) && (r <= 126))
                            _response[i++] = r;

                        while (read(_fd,&r,1) > 0) {
                            if ((r >= 32) && (r <= 126)) {
                                _response[i++] = r;
                                if (i == MAX_STR_LEN)
                                    break;
                            }
                        }

                        _response[i]= '\0';


                        if (okPtr = strstr(_response, "OK"))  { // OK
                            *okPtr = '\0';
                            strcat (_response, " (OK)");
                        }
                        else if (strstr(_response, "ERRO"))      // ERROR
                            status = -1;

                    }
                }
            }

            if (_debug)
                printf ("%s\n", _response);

            return (status);
		}

	public:

        Modem3G(char* port, int bRate, int debug)
        {
            _bRate = bRate;
            strcpy(_port, port);
            _fd = -1;
            _debug = debug;
            if (_debug)
                printf("Modo DEBUG\n");
        }

        ~Modem3G()
        {
            if (_fd > 0) {
                fcntl(_fd, F_GETFL, _oldFlags);
                tcsetattr(_fd, TCSANOW, &_oldOptions);
                close(_fd);
            }
        }

        int modemOpenPort()
		{
			char r;

			_fd = open(_port, O_RDWR | O_NONBLOCK);

			if (_fd == -1)
				return (_fd);

			_oldFlags = fcntl(_fd, F_GETFL, 0);
			if (_oldFlags < 0)
				return (_oldFlags);

			fcntl(_fd, F_SETFL, 0);
			tcgetattr(_fd, &_oldOptions);	                // get the current options for the port

			// Set options for the modem port...

			memset(&_newOptions,0,sizeof(_newOptions));
			_newOptions.c_iflag=0;
			_newOptions.c_oflag=0;
			_newOptions.c_cflag=CS8|CREAD|CLOCAL;           // 8n1, see termios.h for more information
			_newOptions.c_lflag=0;
			_newOptions.c_cc[VMIN]=1;
			_newOptions.c_cc[VTIME]=5;

			cfsetispeed(&_newOptions, _bRate);	            // set baud rates
			cfsetospeed(&_newOptions, _bRate);

			tcsetattr(_fd, TCSANOW, &_newOptions);	        // set the new options for the port...
			fcntl(_fd, F_SETFL, O_NONBLOCK);

			if (_debug)
				printf("Porta %s aberta\n", _port);

			while (read(_fd,&r,1) > 0);
			if (_debug)
				printf("Entrada limpa\n", _port);

			return(0);
		}

        int modemInit()
        {
            char cmd[100];
            int st;
            int atOk = 0;
            int tries = 0;
            const int MAX_TRIES = 100;

            if (_debug)
                printf("Inicializando modem\n");

            while (!atOk && (tries++ < MAX_TRIES)) {
                strcpy(cmd, "AT");
                st = _sendCmd(cmd);
                if (st)
                    return(st);
                if (strstr(_response, "OK"))
                    atOk = -1;
                else {
                    usleep(100000);
//                    strcpy(cmd, "ATZ");	// reset the modem back to default factory settings
//                    st = _sendCmd(cmd);
//                    usleep(100000);
                }
            }

            if (atOk)
                return(0);
            else {
                printf("ERRO: no pude inicializar o modem em %d tentativas\n\n", MAX_TRIES);
                return (-1);
            }
        }

        int getModemInfo()
		{
			char cmd[100];
			int st;

            // Query modem manufacturer
            strcpy(cmd, "AT+CGMI=?");
            st = _sendCmd(cmd);
			if (st)
                printf("ATENO: Modem no aceita que se consulte o nome do seu fabricante\n");
			else {
				strcpy(cmd, "AT+CGMI");			// query modem manufacturer
                st = _sendCmd(cmd);
				if (!st)
					printf("Fabricante do modem: %s\n", _response);
				else
					return(st);
			}

            // Query model number
            strcpy(cmd, "AT+CGMM=?");
            st = _sendCmd(cmd);
			if (st)
                printf("ATENO: Modem no aceita que se consulte o seu modelo\n");
			else {
				strcpy(cmd, "AT+CGMM");		// query model number
                st = _sendCmd(cmd);
				if (!st)
					printf("Modelo do modem: %s\n", _response);
				else
					return(st);
			}

            // Query IMEI number
            strcpy(cmd, "AT+CGSN=?");
            st = _sendCmd(cmd);
			if (st)
                printf("ATENO: Modem no aceita que se consulte o seu IMEI\n");
			else {
				strcpy(cmd, "AT+CGSN");		// query IMEI number
                st = _sendCmd(cmd);
				if (!st)
					printf("Nmero do IMEI: %s\n", _response);
				else
					printf("ERRO: no pude obter IMEI (%s)\n", _response);
			}

            // Query software version
            strcpy(cmd, "AT+CGMR=?");
            st = _sendCmd(cmd);
			if (st)
                printf("ATENO: Modem no aceita que se consulte a verso do software\n");
			else {
				strcpy(cmd, "AT+CGMR");		// query software version
                st = _sendCmd(cmd);
				if (!st)
					printf("Verso do software: %s\n", _response);
				else
					printf("ERRO: no pude obter verso do software (%s)\n", _response);
			}

            // Query activity status
            strcpy(cmd, "AT+CPAS=?");
            st = _sendCmd(cmd);
			if (st)
                printf("ATENO: Modem no aceita que se consulte o status de atividade\n");
			else {
				strcpy(cmd, "AT+CPAS");		// query activity status
                st = _sendCmd(cmd);
				if (!st)
					printf("Status de atividade: %s\n", _response);
				else
					printf("ERRO: no pude obter status de atividade (%s)\n", _response);
			}

            // Query network registration status
            strcpy(cmd, "AT+CREG=?");
            st = _sendCmd(cmd);
			if (st)
                printf("ATENO: Modem no aceita que se consulte o network registration status\n");
			else {
				strcpy(cmd, "AT+CREG");		// query network registration status
                st = _sendCmd(cmd);
				if (!st)
					printf("Network registration status: %s\n", _response);
				else
					printf("ERRO: no pude obter network registration status (%s)\n", _response);
			}

            // Query radio signal strength
            strcpy(cmd, "AT+CSQ=?");
            st = _sendCmd(cmd);
			if (st)
                printf("ATENO: Modem no aceita que se consulte a intensidade do sinal\n");
			else {
				strcpy(cmd, "AT+CSQ");		// query radio signal strength
                st = _sendCmd(cmd);
				if (!st)
					printf("Intensidade do sinal (Received Signal Strength Indicator): %s\n", _response);
				else
					printf("ERRO: no pude obter intensidade do sinal (%s)\n", _response);
			}

            // Check PIN status
            strcpy(cmd, "AT+CPIN?");
            st = _sendCmd(cmd);
			if (!st)
				printf("Status do PIN: %s\n", _response);
			else
				printf("ERRO: no pude obter o status do PIN (%s)\n", _response);

            // Query message center phone number
            strcpy(cmd, "AT+CSCA?");
            st = _sendCmd(cmd);
			if (!st)
				printf("Nmero do Centro de Envio de Mensagens: %s\n", _response);
			else
				printf("ERRO: no pude obter nmero do centro de envio de mensagens (%s)\n", _response);

            // Get current GSM/UMTS network operator, list available operators.
            strcpy(cmd, "AT+COPS?");
            st = _sendCmd(cmd);
			if (!st)
				printf("Operador atual de rede GSM/UMTS: %s\n", _response);
			else
				printf("ERRO: no pude obter o operador atual de rede GSM/UMTS (%s)\n", _response);

            // Query ???
            strcpy(cmd, "AT+CSMP?");
            st = _sendCmd(cmd);
			if (!st)
				printf("CSMP: %s\n", _response);
			else
				printf("ERRO: no pude executar o comando CSMP (%s)\n", _response);

            // Read operator names stored in device memory (hundreds!!!)
//			 strcpy(cmd, "AT+COPN");
//			 st = _send(cmd);
//			 if (!st)
//				 printf("Nome(s) de operadores na memria: %s\n", _response);
//			 else
//				 printf("ERRO: no pude obter o(s) nome(s) de operador(es) na memria (%s)\n", _response);

			return(0);
		}

        int setTextMode()
		{
			char cmd[100];
			int st;

            // Query the available operating modes
            strcpy(cmd, "AT+CMGF=?");
			if (_debug)
				printf("Verificando compatibilidade do modem com o modo texto");
            st = _sendCmd(cmd);
			if (_debug)
				printf("... %s\n", _response);
			if (st)
				return(st);
			if (strstr(_response, "1") == 0) {
                printf("ERRO: modem no compatvel com o modo texto\n\n");
				return (-1);
			}

            // Select the operating mode 1 (SMS text mode)
            strcpy(cmd, "AT+CMGF=1");
			if (_debug)
				printf("Selecionando modo texto\n");
            st = _sendCmd(cmd);
			if (st)
				return(st);
			if (!strstr(_response, "OK"))
				return(-1);

			return(0);
		}

		int setPinNumber(char* pinNumber)
        {
            char dQuote[2];
            char cmd[100];
            int st;

            if (_debug)
                printf("Definindo nmero do PIN\n");

            // Set PIN number
            strcpy(cmd, "AT+CPIN=");
            strcat(cmd, dQuote);
            strcat(cmd, pinNumber);
            strcat(cmd, dQuote);
            st = _sendCmd(cmd);
            if (!st) {
                if (_debug)
                    printf("%s\n", _response);
                return(0);
            }
            else
                printf("ERRO: no pude definir nmero do PIN\n");
            return (-1);
        }

        int getPinStatus()
        {
            char cmd[100];
            int st;

            if (_debug)
                printf("Obtendo status do PIN.\n");

            // Check PIN status
            strcpy(cmd, "AT+CPIN?");
            st = _sendCmd(cmd);
            if (!st) {
                if (_debug)
                    printf("Status do PIN: %s\n", _response);
                if (strstr(_response, "READY") != 0)
                    return(0);
                else
                    return(-1);
            }
            else {
                printf("ERRO: no pude obter o status do PIN\n");
                return (-1);
            }
        }

		int setMessageCenterNumber(char* msgCenterNumber)
		{
			char dQuote[2];
			char telNo[MAX_STR_LEN+1];
			int i, lenPh;
			char cmd[100];
			int st;

			dQuote[0] = '\042'; dQuote[1] = '\000';

            if (_debug)
                printf("Enviando o nmero do centro de mensagens (%s)\n", msgCenterNumber);

            strcpy (telNo, msgCenterNumber);

            // Remove possible CRs/LFs at the end of the phone number
            lenPh = strlen(telNo);
            for (i = (lenPh-1); i >= 0; i--)
                if (*(telNo+i) <= 32)
                    *(telNo+i) = '\0';

            // Set message center number
            strcpy (cmd, "AT+CSCA=");
            strcat (cmd, dQuote);
            strcat (cmd, telNo);
            strcat (cmd, dQuote);

            st = _sendCmd(cmd);
            if (st)
                return(st);
            if (!strstr(_response, "OK"))
                return(-1);

			return(0);
		}

        int setValidity(char* validity)
		{

			char cmd[100];
			int st;
            long int val;
            char* timeType;
            int range = 0;
            int param;

            timeType = strpbrk(validity, "mhdsMHDS");
            if (timeType == 0)
                return (-1);

            val = atol(validity);

            switch (*timeType) { // convert value to minutes
                case 'h':
                case 'H':
                    val *= (long int) 60;
                    break;
                case 'd':
                case 'D':
                    val *= (long int) (60*24);
                    break;
                case 's':
                case 'S':
                    val *= (long int) (60*24*7);
            }

            // Define CSMP parameter value - see http://www.smssolutions.net/tutorials/gsm/sendsmsat/
            if (val < 5) {
                    val = 5;
                    range = 1;
            }
            else if  ((val >= 5) && (val <= 720)) range = 1;
            else if (val < 750) {
                val = 720;
                range = 1;
            }
            else if ((val >= 750) && (val <= 1440)) range = 2;
            else if (val < 2880) {
                val = 1440;
                range = 2;
            }
            else if ((val >= 2880) && (val <= 43200)) range = 3;
            else if (val < 50400) {
                val = 43200;
                range = 3;
            }
            else if ((val >= 50400) && (val <= 635040)) range = 4;
            else {
                val = 635040;
                range = 4;
            }

            switch (range) {
            case 1:
                param = (val / 5) - 1;
                break;
            case 2:
                param = 143 + (val - 12 * 60) / 30;
                break;
            case 3:
                param = (val / (24 * 60)) + 166;
                break;
            case 4:
                param = (val / (7 * 24 * 60)) + 192;
            }

            sprintf(cmd, "AT+CSMP=17,%d,0,0", param);

            /* The following command sets the "flash" mode that
            ONLY exibts the message, without leaving it in the input box
            and the originator's number registered. Some phones
            (e.g. Sony Ericson Xperia) not even
            beep when the message arrives.
            */

            // sprintf(cmd, "AT+CSMP=17,%d,0,16", param);

            st = _sendCmd(cmd);
            if (st)
                return (st);
            if (!strstr(_response, "OK"))
                return(-1);

			return(0);
		}

        int sendMessage(char* phone, char* messageIn)
        {
			char message[2*MAX_STR_LEN];
			char telNo[MAX_STR_LEN+1];
			char dQuote[2], sub[2], crlf[3];
			int i, lenPh;
			char cmd[100];
			int st;
            char* charPtr;
            char fromChr[] =    "";
            char toChr[] =      "AEIOUaeiouAaCcUuAOaoAEOaeo";
            char* toChrPtr;

			dQuote[0] = '\042'; dQuote[1] = '\000';
			sub[0] = '\032'; sub[1] = '\000';
			crlf[0] = '\015'; crlf[1] = '\012'; crlf[2] = '\000';

			strcpy (telNo, phone);

            // Remove possible CRs/LFs at the end of the phone number
			lenPh = strlen(telNo);
			for (i = (lenPh-1); i >= 0; i--)
				if (*(telNo+i) <= 32)
					*(telNo+i) = '\0';

            // Send telno
			strcpy (cmd, "AT+CMGS=");
			strcat (cmd, dQuote);
			strcat (cmd, telNo);
			strcat (cmd, dQuote);
			if (_debug)
				printf("Enviando o nmero do telefone \"%s\" e aguardando o prompt da mensagem\n", telNo);
            st = _sendCmd(cmd);
			if (!strstr(_response, ">")) {
				printf("ERRO: prompt de envio da mensagem no chegou.\n\n");
				return (-1);
			}

            // Convert non 7bit characters
            strcpy (message,  messageIn);
            while (charPtr = strpbrk(message, fromChr)) {
                toChrPtr = strchr(fromChr, *charPtr);
                toChrPtr = (toChrPtr-fromChr) + toChr;
                *charPtr = *toChrPtr;
            }

            // Send message
			if (_debug)
				printf("Enviando a mensagem \"%s\"\n", message);
			strcat (message, sub);
            st = _sendCmd(message);
			if (_debug)
                printf("Status do envio: %s\n", message);   // it reports the message sent instead of "OK"

			return(0);
		}
};

/** Displays application usage.
*/
void showUsage(char** argv)
{
	printf("Uso:\n");
    printf("\t%s {consulta|infomodem|sms|numpin|numcem|validade}\n", argv[0]);
	printf("onde:\n");
    printf("\tconsulta = {-v|--versao} | {-h|--help|?}\n");
    printf("\t\tpara verso ou ajuda (esse texto), e\n");
    printf("\tinfomodem = {-i|--info} {-p|--porta} porta [{-b|--baud} baudrate] [{-d|--debug}]\n");
    printf("\t\tpara obteno das informaes do modem, e\n");
    printf("\tsms = {-m|--mensagem} texto {-p|--porta} porta {-t|tel} tel1[,tel2,...teln] [{-b|--baud} baudrate] [{-d|--debug}]\n");
    printf("\t\tpara o envio de mensagem sms a um ou mais destinatrios, e\n");
    printf("\tnumpin = {-n|--pin} numeropin {-p|--porta} porta [{-b|--baud} baudrate] [{-d|--debug}]\n");
    printf("\t\tpara a definio do nmero do pin para desbloqueio do modem, necessrio, em alguns casos,\n");
    printf("\t\taps a instalao do chip no modem, e\n");
    printf("\tnumcem = {-c|--cem} centroenvio {-p|--porta} porta [{-b|--baud} baudrate] [{-d|--debug}]\n");
    printf("\t\tpara a definio do nmero do centro de envio de mensagens, usualmente j definido no chip, e\n");
    printf("\tvalidade = {-l|--val} tempovalidade {-p|--porta} porta [{-b|--baud} baudrate] [{-d|--debug}]\n");
    printf("\t\tpara a definio do tempo de validade da mensagem no centro de envio de mensagens,\n");
    printf("\t\tcaso o destinatrio no esteja disponvel\n");
	printf("onde:\n");
	printf("\ttexto  o texto da mensagem (se tiver espaos, colocar entre aspas);\n");
	printf("\tporta  o identificador da porta de sada (e.g. /dev/ttyUSB1);\n");
    printf("\ttel  a relao de nmeros de telefones de destino separados por vgulas e sem espaos ou traos\n");
    printf("\t\t(e.g. 99813175,4196601987,98765432);\n");
    printf("\tbaudrate  a taxa de transmisso para comunicao com o modem, em bits por segundo, nos valores\n");
    printf("\t\t2400, 4800, 9600, 19200, ou 38400 (default=9600);\n");
    printf("\tnumeropin  o cdigo de operao do pin, usualmente necessrio quando o chip  inserido do modem;\n");
    printf("\tcentroenvio  o nmero do centro de envio de mensagens;\n");
    printf("\ttempovalidade  tempo que o centro de envio guarda a mensagem. As formas aceitveis para o valor\n");
    printf("\t\tdo tempo de validade so nM, nH, nD ou nS, onde \'n\'  o nmero aproximado de minutos, horas,\n");
    printf("\t\tdias ou semanas, respectivamente;\n");
    printf("\tse -d ou --debug so informados, sero exibidos os comandos enviados ao modem durante a execuo.\n");
	printf("\n");
};

int main(int argc, char** argv)
{

    /** Short versions of the command line parameters.*/
    char short_options [] = "p:b:m:t:c:n:l:idvh?";

    /** Long versions of the command line parameters.*/
	struct option long_options [] = {
  		{ "porta",		required_argument,		NULL,			'p' },
		{ "baud",		required_argument,		NULL,			'b' },
		{ "mensagem",	required_argument,		NULL,			'm' },
		{ "tel",		required_argument,		NULL,			't' },
        { "cem",		required_argument,		NULL,			'c' },
        { "pin",		required_argument,		NULL,			'n' },
        { "val",		required_argument,		NULL,			'l' },
        { "info",		no_argument,			NULL,			'i' },
		{ "debug",		no_argument,			NULL,			'd' },
		{ "versao",		no_argument,			NULL,			'v' },
  		{ "help",		no_argument,			NULL,			'h' },
  		{ 0, 			0, 						0, 				0 }
	};
	char port[MAX_STR_LEN+1];
	char message[MAX_MSG_LEN+1];
	char phone[MAX_STR_LEN+1];
	char sca[MAX_STR_LEN+1];
    char pinNumber[MAX_STR_LEN+1];
	char baud[MAX_STR_LEN+1];
    char validity[MAX_STR_LEN+1];
	int bRate = B9600;
	char *ph, *comma;
	int debug = 0;
	int setSCA = 0;
    int info = 0;
    int pin = 0;
    int version = 0;
    int help = 0;
	int msgToBeSent = 0;
    int val = 0;

	// Initialize variables
	port[0] ='\0';
    message[0] ='\0';
	phone[0] ='\0';
    sca[0] ='\0';
    pinNumber[0] ='\0';
	baud[0] ='\0';
    validity[0] ='\0';

    // Get and verify command line parameters
	for (;;) {
		int index;
		int c;

		c = getopt_long (argc, argv,
			short_options, long_options,
			&index);

        if (-1 == c) {
			break;
        }

		switch (c) {
			case 0: 		//getopt_long() flag
				break;

			case 'p':
				if (strlen(optarg) > MAX_STR_LEN) {
					printf("ERRO: parmetro \'%c\' de comprimento superior a %d caracteres\n\n", c, MAX_STR_LEN);
					return (-1);
				}
				strcpy (port, optarg);
				break;

			case 'b':
				if (strlen(optarg) > MAX_STR_LEN) {
					printf("ERRO: parmetro \'%c\' de comprimento superior a %d caracteres\n\n", c, MAX_STR_LEN);
					return (-1);
				}
				strcpy (baud, optarg);
				break;

			case 'm':
				if (strlen(optarg) > MAX_MSG_LEN) {
					printf("ERRO: mensagem \'%c\' de comprimento superior a %d caracteres\n\n", c, MAX_MSG_LEN);
					return (-1);
				}
				strcpy (message, optarg);
				msgToBeSent = -1;
				break;

			case 't':
				if (strlen(optarg) > MAX_STR_LEN) {
					printf("ERRO: parmetro \'%c\' de comprimento superior a %d caracteres\n\n", c, MAX_STR_LEN);
					return (-1);
				}
				strcpy (phone, optarg);
				break;

            case 'c':
				if (strlen(optarg) > MAX_STR_LEN) {
					printf("ERRO: parmetro \'%c\' de comprimento superior a %d caracteres\n\n", c, MAX_STR_LEN);
					return (-1);
				}
				strcpy (sca, optarg);
				setSCA = -1;
				break;

            case 'n':
                if (strlen(optarg) > MAX_STR_LEN) {
                    printf("ERRO: parmetro \'%c\' de comprimento superior a %d caracteres\n\n", c, MAX_STR_LEN);
                    return (-1);
                }
                strcpy (pinNumber, optarg);
                pin = -1;
                break;

            case 'l':
                if (strlen(optarg) > MAX_STR_LEN) {
                    printf("ERRO: parmetro \'%c\' de comprimento superior a %d caracteres\n\n", c, MAX_STR_LEN);
                    return (-1);
                }
                strcpy (validity, optarg);
                val = -1;
                break;

			case 'v':
                version = -1;
                break;

			case 'd':
				debug = -1;
				break;

            case 'i':
				info = -1;
				break;

			case 'h':
			case '?':
                help = -1;
                break;

			default:
				showUsage(argv);
                return (-1);
		}
	}

    if ((val + info + pin + setSCA + msgToBeSent + version + help) == 0) {
        printf("ERRO: uma das aes precisa ser solicitada: consulta, infomodem, sms, validade ou numpin\n\n");
        showUsage(argv);
        return (-1);
    }

    if ((val + info + pin + setSCA + msgToBeSent + version + help) < -1) {
        printf("ERRO: apenas uma das aes pode ser solicitada: consulta, infomodem, sms, validade ou numpin\n\n");
        showUsage(argv);
        return (-1);
    }

    if (info) {
        if (strlen(port) == 0) {
            printf("ERRO: parmetro \'%c\' no fornecido\n\n", 'p');
            showUsage(argv);
            return (-1);
        }
    }

    if (pin) {
        if (strlen(port) == 0) {
            printf("ERRO: parmetro \'%c\' no fornecido\n\n", 'p');
            showUsage(argv);
            return (-1);
        }
    }

    if (setSCA ) {
        if (strlen(port) == 0) {
            printf("ERRO: parmetro \'%c\' no fornecido\n\n", 'p');
            showUsage(argv);
            return (-1);
        }
    }

    if (val ) {
        if (strlen(port) == 0) {
            printf("ERRO: parmetro \'%c\' no fornecido\n\n", 'p');
            showUsage(argv);
            return (-1);
        }
    }

    if (msgToBeSent) {
        if (strlen(phone) == 0) {
            printf("ERRO: parmetro \'%c\' no fornecido\n\n", 't');
            showUsage(argv);
            return (-1);
        }
        if (strlen(port) == 0) {
            printf("ERRO: parmetro \'%c\' no fornecido\n\n", 'p');
            showUsage(argv);
            return (-1);
        }
    }

    if (strlen(baud) != 0) {
        bRate = atoi(baud);
        switch (bRate) {
            case 2400:
                bRate = B2400;
                break;
            case 4800:
                bRate = B4800;
                break;
            case 9600:
                bRate = B9600;
                break;
            case 19200:
                bRate = B19200;
                break;
            case 38400:
                bRate = B38400;
                break;
            default:
                printf("ERRO: parmetro \'%c\' incorreto\n\n", 'b');
                showUsage(argv);
                return(-1);
        }
    }

    // Process command

    if (version) {
        printf("Verso: %s\n\n", VERSION);
        return(0);
    }

    if (help) {
        showUsage(argv);
        return (0);
    }

    Modem3G modem(port, bRate, debug);

    if (modem.modemOpenPort()) {
		printf("ERRO: no pude abrir porta %s\n\n", port);
		return (-1);
	}

	if (info) {
        if (modem.modemInit()) {
            return(-1);
        }
        return (modem.getModemInfo());
	}

    if (pin) {
        if (modem.modemInit()) {
            return(-1);
        }
        if (modem.getPinStatus() == 0) {
            printf("ERRO: o chip j se encontra no status READY\n\n");
            return (-1);
        }
        if (modem.setPinNumber(pinNumber)) {
            printf("ERRO: no pude definir o nmero do pin\n\n");
            return (-1);
        }
        return (0);
    }

    if (setSCA) {
        if (modem.modemInit()) {
            return(-1);
        }
        if (modem.setMessageCenterNumber(sca)) {
            printf("ERRO: no pude definir o nmero do centro de envio de mensagens\n\n");
            return (-1);
        }
        return (0);
    }

    if (val) {
        if (modem.modemInit()) {
            return(-1);
        }
        if (modem.setValidity(validity)) {
            printf("ERRO: no pude definir o tempo de validade da mensagem\n\n");
            return (-1);
        }
        return (0);
    }

	/* Send message to all recipients */
    if (msgToBeSent) {
        // Check pin
        if (modem.modemInit())
            return (-1);
        if (modem.getPinStatus() != 0) {
            printf("ERRO: o pin no foi definido ainda. Execute a opo \"numpin\"\n\n");
            showUsage(argv);
            return (-1);
        }
        int error = 0;
        strcat(phone, ",");
        ph = phone;
        while (strlen(ph) != 0) {
            if (modem.modemInit())
                error--;
            else {
                if (modem.setTextMode())
                    error--;
                comma = strstr(ph, ",");
                *comma = '\0';
                if (!error && (strlen(ph)!=0))
                    if (modem.sendMessage(ph, message))
                        error--;
            }
            if (error)
                printf("ERRO: no pude enviar mensagem \"%s\" para o nmero \"%s\"\n\n", message, ph);
            ph = comma+1;
            usleep(500000);
        }
        return (error);
    }

	return(0);
}
