/** sendsms: sends an sms via gsm phone/modem.
* Compilation:  g++ -o sendsms sendsms.c++
* \author L.A.M.P.
* \version V1.0 - 2012.02.05
*/

#define VERSION "V1.0 - 2012.02.05"
#define MAX_LEN 255

// C includes
#include <getopt.h>		/* getopt_long() */
#include <stdio.h>		/* Standard input/output definitions */
#include <stdlib.h>		/* Standard library definitions */
#include <string.h>		/* String function definitions */
#include <unistd.h>		/* UNIX standard function definitions */
#include <fcntl.h>		/* File control definitions */
#include <errno.h>		/* Error number definitions */
#include <termios.h>	/* POSIX terminal control definitions */

class SerialPort
{
	private:
	
		struct termios _newOptions;
		struct termios _oldOptions;
		int _bRate;
		char _port[MAX_LEN+1];
		int _fd;
		int _oldFlags;
		
		int _send (char* messageIn)
		{
			char msg[MAX_LEN+1];
			char crlf[3];
			int lenMsg;
			
			crlf[0] = '\015';
			crlf[1] = '\012';
			crlf[2] = '\000';
			
			strcpy (msg, messageIn);
			strcat (msg, crlf);
			
			lenMsg = strlen(msg);
			
			if (write(_fd, msg, lenMsg) != lenMsg)	// send message
				return (-1);
			
			return(0);
		}
		
	public:
	
		SerialPort(char* port, int bRate)
		{
			_bRate = bRate;
			strcpy(_port, port);
			_fd = -1;
		}
		
		int fOpen()
		{	
			_fd = open(_port, O_RDWR | O_NOCTTY | O_NDELAY);
			
			if (_fd == -1) 
				return (_fd);
	
			_oldFlags = fcntl(_fd, F_GETFL, 0);
			if (_oldFlags < 0)
				return (_oldFlags);
				
			fcntl(_fd, F_SETFL, 0);
			tcgetattr(_fd, &_oldOptions);	// get the current options for the port
			_newOptions = _oldOptions;
			
			cfsetispeed(&_newOptions, _bRate);	// set baud rates
			cfsetospeed(&_newOptions, _bRate);
			
			/*
			* Enable the receiver and set local mode...
			*/

			/* set raw input, 1 second timeout */
			_newOptions.c_cflag |= (CRTSCTS | CS8 | 0 | 0 | 0 | CLOCAL | CREAD);
			_newOptions.c_lflag     &= ~(ICANON | ECHO | ECHOE | ISIG);
			_newOptions.c_oflag     &= ~OPOST;
			_newOptions.c_cc[VMIN]  = 0;
			_newOptions.c_cc[VTIME] = 10;

			tcsetattr(_fd, TCSANOW, &_newOptions);	// set the new options for the port...
			
			return(0);
		}
		
		int mSend(char* phone, char* messageIn)
		{
			char message[MAX_LEN+1];
			char telNo[MAX_LEN+1];
			char dQuote[2], sub[2];
			int i, lenPh;
			
			dQuote[0] = '\042'; dQuote[1] = '\000';
			sub[0] = '\032'; sub[1] = '\000';
			
			if (_send("AT")  != 0)
				return(-1);			
			
			usleep(500000);
			
			if (_send("ATZ"))
				return(-1);
			
			usleep(500000);
			
			if (_send("AT+CMGF=1"))
				return(-1);
				
			usleep(500000);
			
			strcpy (telNo, phone);
			
			/* Remove possible CRs/LFs at the end of the phone number */
			lenPh = strlen(telNo);
			for (i = (lenPh-1); i >= 0; i--)
				if (*(telNo+i) < 32)
					*(telNo+i) = '\0';
			
			strcpy (message, "AT+CMGS=");
			strcat (message, dQuote);
			strcat (message, telNo);
			strcat (message, dQuote);
			if (_send(message))
				return(-1);
				
			usleep(1000000);
			
			strcpy (message,  messageIn);
			strcat (message, sub);
			if (_send(message))
				return(-1);
				
			usleep(2000000);
			
			return(0);
		}
				
		~SerialPort()
		{
			if (_fd > 0) {
				fcntl(_fd, F_GETFL, _oldFlags);
				tcsetattr(_fd, TCSANOW, &_oldOptions);
				close(_fd);
			}
		}
		
};

/** Displays application usage.
*/
void showUsage(char** argv)
{
	printf("Uso:\n");
	printf("\t%s {consulta|sms}\n", argv[0]);
	printf("onde:\n");
	printf("\tconsulta = {-v|--versao} | {-h|--help|?}, para verso ou ajuda (esse texto), e\n");
	printf("\tsms = {-m|--mensagem} texto {-p|--porta} porta {-t|tel} tel1[,tel2,...teln] [{-b|--baud} baudrate]\n");
	printf("onde:\n");
	printf("\ttexto  o texto da mensagem (se tiver espaos, colocar entre aspas);\n");
	printf("\tporta  o identificador da porta de sada (e.g. /dev/ttyUSB1);\n");
	printf("\ttel  a relao de nmeros de telefones de destino separados por vgulas e sem espaos ou traos (e.g. 99813175,4196601987,98765432);\n");
	printf("\tbaudrate = {2400|4800|9600|19200|38400}  a velocidade da porta (default=9600).\n");
	printf("\n");
};
	
int main(int argc, char** argv)
{

	/** Short versions of the command line parameters. 
	*/
	char short_options [] = "p:b:m:t:vh?";	
	/** Long versions of the command line parameters.
	*/
	struct option long_options [] = {	 
  		{ "porta",		required_argument,		NULL,			'p' },
		{ "baud",		required_argument,		NULL,			'b' },
		{ "mensagem",	required_argument,		NULL,			'm' },
		{ "tel",		required_argument,		NULL,			't' },
		{ "versao",		no_argument,			NULL,			'v' },
  		{ "help",		no_argument,			NULL,			'h' },
  		{ 0, 			0, 						0, 				0 }
	}; 
	char port[MAX_LEN+1];
	char message[MAX_LEN+1];
	char phone[MAX_LEN+1];
	char baud[MAX_LEN+1];
	int bRate = B9600;
	char *ph, *comma;
	
	// Initialize variables
	port[0] ='\0';
	message[0] ='\0';
	phone[0] ='\0';
	baud[0] ='\0';
	
	// Get and process command line parameters
	for (;;) {
		int index;
		int c;
		
		c = getopt_long (argc, argv,
			short_options, long_options,
			&index);
			
		if (-1 == c) 
			break;
		
		switch (c) {
			case 0: 		//getopt_long() flag
				break;
				
			case 'p':
				if (strlen(optarg) > MAX_LEN) {
					printf("Erro: parmetro \'%c\' de comprimento superior a %d caracteres\n\n", c, MAX_LEN);
					return (-1);
				}
				strcpy (port, optarg);
				break;
				
			case 'b':
				if (strlen(optarg) > MAX_LEN) {
					printf("Erro: parmetro \'%c\' de comprimento superior a %d caracteres\n\n", c, MAX_LEN);
					return (-1);
				}
				strcpy (baud, optarg);
				break;
				
			case 'm':
				if (strlen(optarg) > MAX_LEN) {
					printf("Erro: parmetro \'%c\' de comprimento superior a %d caracteres\n\n", c, MAX_LEN);
					return (-1);
				}
				strcpy (message, optarg);
				break;
			
			case 't':
				if (strlen(optarg) > MAX_LEN) {
					printf("Erro: parmetro \'%c\' de comprimento superior a %d caracteres\n\n", c, MAX_LEN);
					return (-1);
				}
				strcpy (phone, optarg);
				break;
				
			case 'v':
				printf("Verso: %s\n\n", VERSION);
				return(0);
				
			case 'h':
			case '?':
				showUsage(argv);
				return (-1);
	
			default:
				showUsage(argv);
				return (0);
		}
	}
	
	if (strlen(port) == 0) {
		printf("Erro: parmetro \'%c\' no fornecido\n\n", 'p');
		return (-1);
	}
	
	if (strlen(message) == 0) {
		printf("Erro: parmetro \'%c\' no fornecido\n\n", 'm');
		return (-1);
	}
	
	if (strlen(phone) == 0) {
		printf("Erro: parmetro \'%c\' no fornecido\n\n", 't');
		return (-1);
	}
	
	if (strlen(baud) != 0) {
		bRate = atoi(baud);
		switch (bRate) {
			case 2400:
				bRate = B2400;
				break;
			case 4800:
				bRate = B4800;
				break;
			case 9600:
				bRate = B9600;
				break;
			case 19200:
				bRate = B19200;
				break;
			case 38400:
				bRate = B38400;
				break;
			default:
				printf("Erro: parmetro \'%c\' incorreto\n\n", 'b');
				return(-1);
		}
	}
	
	SerialPort p(port, bRate);
	
	if (p.fOpen()) 
		printf("Erro: no pude abrir porta %s\n\n", port);
	
	/* Send message to all recipients */
	strcat(phone, ",");
	ph = phone;
	while (strlen(ph) != 0) {
		comma = strstr(ph, ",");
		*comma = '\0';
		if (strlen(ph)) {
			//printf ("Enviando para %s\n", ph);
			if (p.mSend(ph, message)) {
				printf("Erro: no pude enviar mensagem %s para o nmero %s\n\n", message, ph);
				return (-1);
			}	
		}
		ph = comma+1;
	}
}
